<?php
/**
 * WooCommerce Constant Contact
 *
 * This source file is subject to the GNU General Public License v3.0
 * that is bundled with this package in the file license.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.html
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@skyverge.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade WooCommerce Constant Contact to newer
 * versions in the future. If you wish to customize WooCommerce Constant Contact for your
 * needs please refer to http://docs.woothemes.com/document/woocommerce-constant-contact/ for more information.
 *
 * @package     WC-Constant-Contact/Classes
 * @author      SkyVerge
 * @copyright   Copyright (c) 2013-2014, SkyVerge, Inc.
 * @license     http://www.gnu.org/licenses/gpl-3.0.html GNU General Public License v3.0
 */

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * A simple widget for displaying a signup for the admin-selected constant contact email list
 *
 * @since 1.0
 * @extends \WP_Widget
 */
class WC_Constant_Contact_Widget extends WP_Widget {


	/**
	 * Setup the widget options
	 *
	 * @since 1.0
	 */
	public function __construct() {

		// set widget options
		$options = array(
			'classname'   => 'widget_wc_constant_contact',
			'description' => __( 'Allow your customers to subscribe to your Constant Contact email list.', WC_Constant_Contact::TEXT_DOMAIN ),
		);

		// instantiate the widget
		parent::__construct( 'wc_constant_contact', __( 'WooCommerce Constant Contact', WC_Constant_Contact::TEXT_DOMAIN ), $options );

		// add AJAX if widget is active
		if ( is_active_widget( false, false, $this->id_base ) ) {

			// enqueue js
			add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_widget_js' ) );

			// handle the subscribe form AJAX submit
			add_action( 'wp_ajax_wc_constant_contact_widget_subscribe',        array( $this, 'ajax_process_widget_subscribe' ) );
			add_action( 'wp_ajax_nopriv_wc_constant_contact_widget_subscribe', array( $this, 'ajax_process_widget_subscribe' ) );
		}
	}


	/**
	 * Render the widget
	 *
	 * @since 1.0
	 * @see WP_Widget::widget()
	 * @param array $args
	 * @param array $instance
	 */
	public function widget( $args, $instance ) {
		global $wc_constant_contact;

		if ( ! $wc_constant_contact->get_api() ) {
			return;
		}

		extract( $args );

		$title   = apply_filters( 'widget_title', $instance['title'], $instance, $this->id_base );
		$list_id = $instance['list'];

		echo $before_widget;

		if ( $title ) {
			echo $before_title . $title . $after_title;
		}
		?>
			<form method="post" id="wc_constant_contact_subscribe_widget_form" action="#wc_constant_contact_subscribe_widget_form">
				<div>
					<label class="screen-reader-text hidden" for="s"><?php _e( 'Email Address', WC_Constant_Contact::TEXT_DOMAIN ); ?>:</label>
					<input type="hidden" id="wc_constant_contact_subscribe_list_id" value="<?php echo $list_id; ?>" />
					<input type="text" name="wc_constant_contact_subscribe_email" id="wc_constant_contact_subscribe_email" placeholder="<?php _e('Your email address', WC_Constant_Contact::TEXT_DOMAIN ); ?>" />
					<input type="submit" id="wc_constant_contact_subscribe" value="<?php _e( 'Subscribe', WC_Constant_Contact::TEXT_DOMAIN ); ?>" />
				</div>
			</form>
		<?php

		echo $after_widget;
	}


	/**
	 * Update the widget title & selected email list
	 *
	 * @since 1.0
	 * @see WP_Widget::update()
	 * @param array $new_instance
	 * @param array $old_instance
	 * @return array
	 */
	public function update( $new_instance, $old_instance ) {

		$instance['title'] = strip_tags( stripslashes( $new_instance['title'] ) );
		$instance['list']  = strip_tags( stripslashes( $new_instance['list'] ) );

		return $instance;
	}


	/**
	 * Render the admin form for the widget
	 *
	 * @since 1.0
	 * @see WP_Widget::form()
	 * @param array $instance
	 * @return void
	 */
	public function form( $instance ) {
		global $wc_constant_contact;

		if ( ! $wc_constant_contact->get_api() ) {
			?><p><?php _e( 'You must enter your API info in  WooCommerce > Settings > Constant Contact before using this widget.', WC_Constant_Contact::TEXT_DOMAIN ); ?></p><?php
			return;
		}

		try {

			$lists = array_merge( array( '' => '' ), $wc_constant_contact->get_api()->get_lists() );

		} catch ( SV_WC_API_Exception $e ) {

			$lists = array( '' => sprintf( __( 'Oops, something went wrong: %s', WC_Constant_Contact::TEXT_DOMAIN ), $e->getMessage() ) );
		}

		?>
			<p><label for="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>"><?php _e( 'Title', WC_Constant_Contact::TEXT_DOMAIN ) ?>:</label>
				<input type="text" class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'title' ) ); ?>" value="<?php echo esc_attr( ( isset ( $instance['title'] ) ) ? $instance['title'] : __( 'Email List', WC_Constant_Contact::TEXT_DOMAIN ) ); ?>" /></p>
			<p><label for="<?php echo esc_attr( $this->get_field_id( 'list' ) ); ?>"><?php _e( 'Email List', WC_Constant_Contact::TEXT_DOMAIN ); ?></label>
				<?php
				echo '<select id="' . esc_attr( $this->get_field_id( 'list' ) ) .'" name="' . esc_attr( $this->get_field_name( 'list' ) ) .'" class="widefat">';
				if ( ! empty( $lists ) ) :
					foreach ( $lists as $list_id => $list_name ) :
						?><option value="<?php echo esc_attr( $list_id ); ?>" <?php selected( ( isset( $instance['list'] ) ) ? $instance['list'] : '', $list_id ); ?>><?php echo esc_html( $list_name ); ?></option><?php
					endforeach;
				endif;

				echo '</select>';

				echo '<small>' . __( 'Select an email list to subscribe customers to or leave blank to use the list defined under WooCommerce > Settings > Constant Contact.', WC_Constant_Contact::TEXT_DOMAIN ) . '</small>';
				?>
			</p>
		<?php
	}


	/**
	 * Enqueue the widget JS
	 *
	 * @since 1.3
	 */
	public function enqueue_widget_js() {

		$nonce    = wp_create_nonce( 'wc_constant_contact' );

		wc_enqueue_js( '
			/* Constant Contact AJAX Widget Subscribe */
			$( "#wc_constant_contact_subscribe_widget_form" ).submit( function( e ) {

				e.preventDefault();

				var $form = $( this );

				if ( $form.is( ".processing" ) ) return false;

				$form.addClass( "processing" ).block( { message: null, overlayCSS: { background: "#fff url(" + woocommerce_params.ajax_loader_url + ") no-repeat center", backgroundSize: "16px 16px", opacity: 0.6 } } );

				var data = {
					action:   "wc_constant_contact_widget_subscribe",
					security: "' . $nonce . '",
					email:    $( "#wc_constant_contact_subscribe_email" ).val(),
					list_id:  $( "#wc_constant_contact_subscribe_list_id" ).val()
				};

				$.ajax({
					type:     "POST",
					url:      woocommerce_params.ajax_url,
					data:     data,
					dataType: "json",
					success:  function( response ) {

						$form.removeClass( "processing" ).unblock();

						// remove any previous messages
						$form.prev( ".woocommerce" ).remove();

						// show messages
						$form.before( response.data );

						if ( response.success ) {
							$form.remove();
						}
					}
				});
				return false;
			});
		' );
	}


	/**
	 * Process the widget AJAX subscribe
	 *
	 * @since 1.3
	 */
	public function ajax_process_widget_subscribe() {
		global $wc_constant_contact;

		// security check
		check_ajax_referer( 'wc_constant_contact', 'security' );

		$email   = ( ! empty( $_POST['email'] ) ) ? wc_clean( $_POST['email'] ) : '';
		$list_id = ( ! empty( $_POST['list_id'] ) ) ?  $_POST['list_id'] : null;

		if ( ! is_email( $email ) ) {

			$error = '<div class="woocommerce"><div class="woocommerce-error">' . __( 'Please enter a valid email address.', WC_Constant_Contact::TEXT_DOMAIN ) . '</div></div>';

			wp_send_json_error( $error );

		} else {

			try {

				$wc_constant_contact->get_api()->subscribe( $email, $list_id );

				$success = '<div class="woocommerce"><div class="woocommerce-message">' . __( 'Thanks for subscribing.', WC_Constant_Contact::TEXT_DOMAIN ) . '</div></div>';

				wp_send_json_success( $success );

			} catch ( SV_WC_API_Exception $e ) {

				$error = '<div class="woocommerce"><div class="woocommerce-error">' . __( 'Oops, something went wrong. Please try again later.', WC_Constant_Contact::TEXT_DOMAIN ) . '</div>';

				wp_send_json_error( $error );

				$wc_constant_contact->log( sprintf( __( 'Widget Signup: %s', WC_Constant_Contact::TEXT_DOMAIN ), $e->getMessage() ) );
			}
		}
	}


} // end \WC_Constant_Contact_Widget class
